#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.

import os.path
import shutil
import sys
from os import listdir
from os.path import dirname, abspath, join, exists
from shutil import copy

# If DEBUG = False, it will override existing configuration folders
DEBUG = False

# Useful folders
SCRIPT_FOLDER = dirname(abspath(__file__))
GENERATED_CFG_FOLDER = join(SCRIPT_FOLDER, 'cfg')
ROOT_FOLDER = dirname(dirname(dirname(dirname(SCRIPT_FOLDER))))

# Destinations folders
ETC_SHINKEN = join(ROOT_FOLDER, 'sources', 'python3', 'etc', 'shinkensolutions')

SOURCES_MODULES = join(ETC_SHINKEN, 'modules')
SOURCES_MODULES_SOURCES = join(ETC_SHINKEN, 'sources')
SOURCES_MODULES_LISTENERS = join(ETC_SHINKEN, 'listeners')
SOURCES_DEFAULT_MODULES = join(ETC_SHINKEN, '_default', 'daemons')
ELEMENTS_DEFAULT_PROPERTIES = join(ETC_SHINKEN, '_default', 'elements_default_properties')
TOOLS_USED_BY_SHINKEN = join(ETC_SHINKEN, 'tools_used_by_shinken')

SKELETONS = join(ROOT_FOLDER, 'tarball', 'shinken-enterprise-tarball', 'skeletons')
SKELETONS_ARBITER_MODULES = join(SKELETONS, 'configuration', 'daemons', 'arbiters', 'modules')
SKELETONS_BROKER_MODULES = join(SKELETONS, 'configuration', 'daemons', 'brokers', 'modules')
SKELETONS_SCHEDULERS_MODULES = join(SKELETONS, 'configuration', 'daemons', 'schedulers', 'modules')
SKELETONS_SYNCHRONIZER = join(SKELETONS, 'configuration', 'daemons', 'synchronizers')
SKELETONS_SYNCHRONIZER_SOURCES = join(SKELETONS_SYNCHRONIZER, 'sources')
SKELETONS_ELEMENTS_DEFAULT_PROPERTIES = join(SKELETONS, 'configuration', 'monitoring_policy', 'elements', 'elements_default_properties')

SKELETONS_EXAMPLES = join(SKELETONS, 'examples')
SKELETONS_TOOLS_USED_BY_SHINKEN_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'tools_used_by_shinken')
SKELETONS_BROKER_MODULES_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'daemons', 'brokers', 'modules')
SKELETONS_POLLER_MODULES_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'daemons', 'pollers', 'modules')
SKELETONS_REACTIONNER_MODULES_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'daemons', 'reactionners', 'modules')
SKELETONS_RECEIVER_MODULES_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'daemons', 'receivers', 'modules')
SKELETONS_SCHEDULER_MODULES_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'daemons', 'schedulers', 'modules')
SKELETONS_ARBITER_MODULES_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'daemons', 'arbiters', 'modules')
SKELETONS_SYNCHRONIZER_SOURCES_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'daemons', 'synchronizers', 'sources')
SKELETONS_SYNCHRONIZER_LISTENERS_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'daemons', 'synchronizers', 'listeners')
SKELETONS_ELEMENTS_DEFAULT_PROPERTIES_EXAMPLES = join(SKELETONS_EXAMPLES, 'configuration', 'monitoring_policy', 'elements', 'elements_default_properties')

def _get_destination_file_name(file_name):
    return file_name if not DEBUG else '%s.DEBUG' % file_name


# Change this to add / remove destination folders
CFG_FILE_DEST_MAPPING = {
    'default_properties_configuration.cfg'                                : [
        join(ELEMENTS_DEFAULT_PROPERTIES, _get_destination_file_name('default_properties_configuration.cfg')),
        join(SKELETONS_ELEMENTS_DEFAULT_PROPERTIES, _get_destination_file_name('default_properties_configuration.cfg')),
        join(SKELETONS_ELEMENTS_DEFAULT_PROPERTIES_EXAMPLES, _get_destination_file_name('default_properties_configuration.cfg'))
    ],
    'architecture-export.cfg'                                         : [
        join(SOURCES_MODULES, _get_destination_file_name('architecture-export.cfg')),
        join(SKELETONS_ARBITER_MODULES_EXAMPLES, 'architecture-export', _get_destination_file_name('architecture-export-example.cfg'))
    ],
    'broker--module-last-configuration-recorder.cfg'                  : [
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'broker--module-last-configuration-recorder', _get_destination_file_name('broker--module-last-configuration-recorder-example.cfg'))
    ],
    'poller--module-last-configuration-recorder.cfg'                  : [
        join(SKELETONS_POLLER_MODULES_EXAMPLES, 'poller--module-last-configuration-recorder', _get_destination_file_name('poller--module-last-configuration-recorder-example.cfg'))
    ],
    'reactionner--module-last-configuration-recorder.cfg'             : [
        join(SKELETONS_REACTIONNER_MODULES_EXAMPLES, 'reactionner--module-last-configuration-recorder', _get_destination_file_name('reactionner--module-last-configuration-recorder-example.cfg'))
    ],
    'receiver--module-last-configuration-recorder.cfg'                : [
        join(SKELETONS_RECEIVER_MODULES_EXAMPLES, 'receiver--module-last-configuration-recorder', _get_destination_file_name('receiver--module-last-configuration-recorder-example.cfg'))
    ],
    'scheduler--module-last-configuration-recorder.cfg'               : [
        join(SKELETONS_SCHEDULER_MODULES_EXAMPLES, 'scheduler--module-last-configuration-recorder', _get_destination_file_name('scheduler--module-last-configuration-recorder-example.cfg'))
    ],
    'synchronizer-import.cfg'                                         : [
        join(SOURCES_MODULES, _get_destination_file_name('synchronizer-import.cfg')),
        join(SKELETONS_ARBITER_MODULES_EXAMPLES, 'synchronizer-import', _get_destination_file_name('synchronizer-import-example.cfg'))
    ],
    'broker-module-livedata.cfg'                                      : [
        join(SOURCES_MODULES, _get_destination_file_name('broker-module-livedata.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'broker-module-livedata', _get_destination_file_name('broker-module-livedata-example.cfg'))
    ],
    'broker--module-report-builder.cfg'                                       : [
        join(SOURCES_MODULES, _get_destination_file_name('broker--module-report-builder.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'broker__module_report_builder', _get_destination_file_name('broker--module-report-builder-example.cfg'))
    ],
    'discovery.cfg'                                                   : [
        join(SOURCES_MODULES_SOURCES, _get_destination_file_name('discovery.cfg')),
        join(SKELETONS_SYNCHRONIZER_SOURCES_EXAMPLES, 'discovery-import', _get_destination_file_name('discovery-example.cfg'))
    ],
    'event_manager_reader.cfg'                                        : [
        join(SOURCES_MODULES, _get_destination_file_name('event_manager_reader.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', 'modules', 'event_manager_reader', _get_destination_file_name('event_manager_reader-example.cfg'))
    ],
    'event_manager_writer.cfg'                                        : [
        join(SOURCES_MODULES, _get_destination_file_name('event_manager_writer.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'event_manager_writer', _get_destination_file_name('event_manager_writer-example.cfg'))
    ],
    'graphite.cfg'                                                    : [
        join(SOURCES_MODULES, _get_destination_file_name('graphite.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'graphite_perfdata', _get_destination_file_name('graphite_perfdata-example.cfg'))
    ],
    'livedata-module-sla-provider.cfg'                                : [
        join(SOURCES_MODULES, _get_destination_file_name('livedata-module-sla-provider.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'broker-module-livedata', 'modules', 'livedata-module-sla-provider', _get_destination_file_name('livedata-module-sla-provider-example.cfg'))
    ],
    'mongodb.cfg'                                                     : [
        join(SOURCES_MODULES, _get_destination_file_name('mongodb.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', 'modules', 'mongodb', _get_destination_file_name('mongodb-example.cfg'))
    ],
    'mongos_socket_keep_alive.cfg'                                    : [
        join(TOOLS_USED_BY_SHINKEN, 'shinken_mongo', _get_destination_file_name('mongos_socket_keep_alive.cfg')),
        join(SKELETONS_TOOLS_USED_BY_SHINKEN_EXAMPLES, 'shinken_mongo', _get_destination_file_name('mongos_socket_keep_alive-example.cfg'))
    ],
    'report-builder--module-sla-reader.cfg'                                       : [
        join(SOURCES_MODULES, _get_destination_file_name('report-builder--module-sla-reader.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'broker__module_report_builder', 'modules', 'report_builder__module_sla_reader', _get_destination_file_name('report-builder--module-sla-reader-example.cfg'))
    ],
    'retention_mongodb.cfg'                                           : [
        join(SOURCES_MODULES, _get_destination_file_name('retention-mongodb.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'retention-mongodb', _get_destination_file_name('retention-mongodb-example.cfg'))
    ],
    'sla.cfg'                                                         : [
        join(SOURCES_MODULES, _get_destination_file_name('sla.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'sla', _get_destination_file_name('sla-example.cfg')),
    ],
    'syncui.cfg'                                                      : [
        join(SKELETONS_SYNCHRONIZER_SOURCES_EXAMPLES, 'syncui', _get_destination_file_name('syncui-example.cfg'))
    ],
    'synchronizer-collector-vmware.cfg'                               : [
        join(SOURCES_MODULES_SOURCES, _get_destination_file_name('synchronizer-collector-vmware.cfg')),
        join(SKELETONS_SYNCHRONIZER_SOURCES_EXAMPLES, 'synchronizer-collector-vmware', _get_destination_file_name('synchronizer-collector-vmware-example.cfg'))
    ],
    'webui.cfg'                                                       : [
        join(SOURCES_MODULES, _get_destination_file_name('webui.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', _get_destination_file_name('webui-example.cfg'))
    ],
    'webui--module-report-handler.cfg'                                        : [
        join(SOURCES_MODULES, _get_destination_file_name('webui--module-report-handler.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', 'modules', 'webui__module_report_handler', _get_destination_file_name('webui--module-report-handler-example.cfg'))
    ],
    'webui_module_service_weather.cfg'                                : [
        join(SOURCES_MODULES, _get_destination_file_name('webui_module_service_weather.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', 'modules', 'webui_module_service_weather', _get_destination_file_name('webui_module_service_weather-example.cfg'))
    ],
    'webui_module_service_weather__default_grid_layout.cfg'           : [
        join(SKELETONS_BROKER_MODULES, 'webui', 'modules', 'webui_module_service_weather', _get_destination_file_name('webui_module_service_weather__default_grid_layout.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', 'modules', 'webui_module_service_weather', _get_destination_file_name('webui_module_service_weather__default_grid_layout-example.cfg'))
    ],
    'webui_module_service_weather__default_info_bar_layout.cfg'       : [
        join(SKELETONS_BROKER_MODULES, 'webui', 'modules', 'webui_module_service_weather', _get_destination_file_name('webui_module_service_weather__default_info_bar_layout.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', 'modules', 'webui_module_service_weather', _get_destination_file_name('webui_module_service_weather__default_info_bar_layout-example.cfg'))
    ],
    'webui_module_service_weather__default_widgets_layout.cfg'        : [
        join(SKELETONS_BROKER_MODULES, 'webui', 'modules', 'webui_module_service_weather', _get_destination_file_name('webui_module_service_weather__default_widgets_layout.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', 'modules', 'webui_module_service_weather', _get_destination_file_name('webui_module_service_weather__default_widgets_layout-example.cfg'))
    ],
    'webui_module_service_weather__default_notifications_settings.cfg': [
        join(SKELETONS_BROKER_MODULES, 'webui', 'modules', 'webui_module_service_weather', _get_destination_file_name('webui_module_service_weather__default_notifications_settings.cfg')),
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', 'modules', 'webui_module_service_weather', _get_destination_file_name('webui_module_service_weather__default_notifications_settings-example.cfg'))
    ],
    'listener-rest.cfg'                                               : [
        join(SOURCES_MODULES_LISTENERS, _get_destination_file_name('listener-rest.cfg')),
        join(SKELETONS_SYNCHRONIZER_LISTENERS_EXAMPLES, 'listener-rest', _get_destination_file_name('listener-rest-example.cfg'))
    ],
    'webui_cfg_overload.cfg'                                          : [
        join(SKELETONS_BROKER_MODULES_EXAMPLES, 'webui', _get_destination_file_name('webui_cfg_overload.cfg')),
        join(SKELETONS_BROKER_MODULES, 'webui', _get_destination_file_name('webui_cfg_overload.cfg'))
    ],
    'webui_default.cfg'                                               : [
        join(SOURCES_DEFAULT_MODULES, 'brokers', 'modules', _get_destination_file_name('webui.cfg')),
    ],
    'base_config_from_synchronizer.cfg'                               : [
        join(ETC_SHINKEN, _get_destination_file_name('synchronizer.cfg')),
    ],
    'base_config_from_synchronizer_cfg_overload.cfg'                  : [
        join(SKELETONS_SYNCHRONIZER, _get_destination_file_name('synchronizer_cfg_overload.cfg')),
    ],
}


def copy_files():
    if not exists(GENERATED_CFG_FOLDER):
        print('━━━━━━━━\033[31mERROR\033[0m : Generated configuration files not found at location :━━━━━━━━\n\t%s' % GENERATED_CFG_FOLDER)
        sys.exit(1)
    for cfg_file in listdir(GENERATED_CFG_FOLDER):
        if cfg_file not in CFG_FILE_DEST_MAPPING:
            continue
        print('Found %s' % cfg_file)
        for destination in CFG_FILE_DEST_MAPPING[cfg_file]:
            source_file = join(GENERATED_CFG_FOLDER, cfg_file)
            print('\t➞ \033[36m%s\033[0m' % destination)
            if destination.endswith('example.cfg') and cfg_file != 'syncui.cfg':
                _create_example_file(source_file, destination)
            else:
                if not os.path.isdir(os.path.dirname(destination)):
                    os.makedirs(os.path.dirname(destination))
                copy(source_file, destination)
        print('')
    print('━━━━━━━━ \033[32mDone\033[0m : Configuration deployment finished ʕ•ᴥ•ʔ ━━━━━━━━')


def _create_example_file(source, destination):
    with open(source, 'r', encoding="utf8") as f:
        source_content = f.readlines()
    
    for idx, line in enumerate(source_content):
        if line.strip().startswith('module_name') or line.strip().startswith('source_name'):
            name = line.strip().split()[1]
            new_name = '%s-example' % name
            source_content[idx] = line.replace(name, new_name)
    
    tmp_destination = '%s.tmp' % destination
    os.makedirs(os.path.dirname(tmp_destination), exist_ok=True)
    
    os.makedirs(os.path.dirname(destination), exist_ok=True)
    
    with open(tmp_destination, 'w+', newline='\n', encoding='utf-8') as f:
        for line in source_content:
            f.write(line)
    
    shutil.move(tmp_destination, destination)


def main():
    if DEBUG:
        print('━━━━━━━━ \033[33mWarning\033[0m : The DEBUG mode is active, destination files will not be overwritten ━━━━━━━━\n')
    copy_files()
