#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2022:
# This file is part of Shinken Enterprise, all rights reserved.
import copy

from shinken.misc.fast_copy import fast_deepcopy
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.pickledb import ShinkenPickleableMeta
from shinkensolutions.common_ui.object.messages import ValidatorMessages
from shinkensolutions.common_ui.object.shinken_ui_composite.shinken_ui_property_bad_value import ShinkenUIPropertyBadValue

if TYPE_CHECKING:
    from shinken.misc.type_hint import TYPE_CHECKING, Dict, List, Any, Union, Type, Tuple, TypeVar, Optional
    
    _AbstractShinkenUICompositeVar = TypeVar('_AbstractShinkenUICompositeVar', bound='AbstractShinkenUIComposite')  # noqa: PyCharm should accept a string for 'bound' parameter

ROOT_ASSOCIATED_KEY = '__ROOT__'


class AbstractShinkenUIComposite(metaclass=ShinkenPickleableMeta):
    __slots__ = ('_source_value', 'associated_key', 'validator_messages', 'validator_message_type', 'mandatory')
    
    
    def __init__(self, associated_key, validator_message_type, mandatory):
        # type: (str, Type[ValidatorMessages], bool) -> None
        self._source_value = None  # type: Any
        self.associated_key = associated_key
        if validator_message_type is None:
            validator_message_type = ValidatorMessages
        self.validator_message_type = validator_message_type
        self.validator_messages = validator_message_type(None)
        self.mandatory = mandatory
    
    
    def copy(self):
        # type: (_AbstractShinkenUICompositeVar) -> _AbstractShinkenUICompositeVar
        # Must be the default deepcopy() method because fast_deepcopy() use its own dispatcher, and we do not want to register *ALL* objects and *ALL* properties
        # Therefore we use __reduce__ :)
        return copy.deepcopy(self)
    
    
    def _get_associated_value_from_list_or_dict(self, _value, associated_key_steps, default=None):
        # type: (Union[Dict, List], List, Any) -> Any
        if len(associated_key_steps) == 0:
            return _value
        
        if isinstance(_value, list):
            return self._get_associated_value_from_list(_value, associated_key_steps, default)
        elif isinstance(_value, dict):
            return self._get_associated_value_from_dictionary(_value, associated_key_steps, default)
        return default
    
    
    def _get_associated_value_from_list(self, _list, associated_key_steps, default=None):
        # type: (List, List, Any) -> Any
        try:
            _list_idx = int(associated_key_steps[0])
        except ValueError:
            return default
        
        try:
            _value = _list[_list_idx]
        except (TypeError, KeyError):
            return default
        
        return self._get_associated_value_from_list_or_dict(_value, associated_key_steps[1:], default)
    
    
    def _get_associated_value_from_dictionary(self, dictionary, associated_key_steps, default=None):
        # type: (Dict, List, Any) -> Any
        try:
            _value = dictionary[associated_key_steps[0]]
        except (TypeError, KeyError):
            return default
        
        return self._get_associated_value_from_list_or_dict(_value, associated_key_steps[1:], default)
    
    
    def _get_associated_value(self, source, default=None):
        if self.associated_key == ROOT_ASSOCIATED_KEY:
            return fast_deepcopy(source)
        _associated_key_steps = self.associated_key.split('.')
        _value = source
        return self._get_associated_value_from_list_or_dict(_value, _associated_key_steps, default)
    
    
    def _pop_associated_value(self, source):
        if self.associated_key == ROOT_ASSOCIATED_KEY:
            return
        _associated_key_steps, _, _last_key_step = self.associated_key.rpartition('.')
        _value = source
        if _associated_key_steps:
            _associated_key_steps = _associated_key_steps.split('.')
            _value = self._get_associated_value_from_list_or_dict(source, _associated_key_steps)
        if _value is not None:
            if isinstance(_value, list):
                try:
                    _list_idx = int(_last_key_step)
                except ValueError:
                    return
                # _value.pop(_list_idx)
            elif isinstance(_value, dict):
                _value.pop(_last_key_step, None)
    
    
    def _handle_bad_value(self, validation_status, trad_key, trad_args):
        pass
    
    
    def add_trad_message(self, validation_status, trad_key, trad_args=(), list_keys=None):
        # type: (str, str, Tuple, Optional[Union[tuple,str]]) -> None
        self.validator_messages.add_trad_message(
            validation_status,
            trad_key,
            trad_args,
            list_keys=list_keys
        )
    
    
    def add_validation_issue(self, validation_status, trad_key, trad_args=(), list_keys=None):
        # type: (str, str, Tuple, Optional[Union[tuple,str]])-> None
        # This function will generate a BadInheritedValue to avoid validation on children if key and value are given
        # Then we add a translated message on the validated object
        self.add_trad_message(
            validation_status,
            trad_key,
            trad_args,
            list_keys=list_keys
        )
        self._handle_bad_value(validation_status, trad_key, trad_args)
    
    
    def _set_source_value_from_source(self, source):
        raise NotImplementedError()
    
    
    def set_source_value_from_source(self, source):
        self._set_source_value_from_source(source)
        self._pop_associated_value(source)
    
    
    def to_flattened_format(self, required_value):  # All available values is in SHINKEN_UI_VALUE class. Better type hint will be set up when enums will be available :')
        # type: (str) -> Any
        raise NotImplementedError()
    
    
    # Must recursively call to_dict() method of underlying ShinkenUIComposite objects
    def _get_stored_values_to_dict(self, computed_value, source_value, config_value, include_validation_messages):
        # type: (bool, bool, bool, bool) -> Dict[str, Any]
        raise NotImplementedError()
    
    
    def to_dict(self, computed_value=False, source_value=False, config_value=False, include_validation_messages=True):
        # type: (bool, bool, bool, bool) -> Dict[str, Any]
        if not any((computed_value, source_value, config_value)):
            raise TypeError('Invalid arguments. Please set to True the values you want')
        _dict_values = self._get_stored_values_to_dict(computed_value=computed_value, source_value=source_value, config_value=config_value, include_validation_messages=include_validation_messages)
        return _dict_values
    
    
    def apply_config_value_to_computed_if_needed(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def apply_source_value_to_computed_if_needed(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def apply_config_or_source_value_to_computed_if_needed(self):
        # type: () -> None
        raise NotImplementedError()
    
    
    def replace_properties_default_values_from(self, other):
        # type: (Any) -> None
        raise NotImplementedError()
    
    
    def has_critical_errors(self):
        # type: () -> bool
        return self.validator_messages.has_critical()
    
    
    def has_errors(self):
        # type: () -> bool
        return self.validator_messages.has_error()
    
    
    def has_warnings(self):
        # type: () -> bool
        return self.validator_messages.has_warning()
    
    
    def get_critical_errors_nb(self):
        # type: () -> int
        return self.validator_messages.get_nb_critical()
    
    
    def get_errors_nb(self):
        # type: () -> int
        return self.validator_messages.get_nb_error()
    
    
    def get_warnings_nb(self):
        # type: () -> int
        return self.validator_messages.get_nb_warning()
    
    
    def get_value(self, default=None):
        # type: (Any) -> Any
        value = self.get_source_value()
        return value if value is not None else default
    
    
    def get_source_value(self):
        # type: () -> Any
        return self._source_value
    
    
    def get_critical_copy(self):
        # type: () -> List
        return self.validator_messages.get_critical_copy()
    
    
    def get_errors_copy(self):
        # type: () -> List
        return self.validator_messages.get_errors_copy()
    
    
    def get_warnings_copy(self):
        # type: () -> List
        return self.validator_messages.get_warnings_copy()
    
    
    def is_source_missing(self):
        # type: () -> bool
        raise NotImplementedError()
    
    
    def is_source_valid(self):
        # type: () -> bool
        return not isinstance(self._source_value, ShinkenUIPropertyBadValue)
    
    
    def is_source_incorrect(self):
        # type: () -> bool
        return isinstance(self._source_value, ShinkenUIPropertyBadValue)
    
    
    def is_source_incorrect_or_missing(self):
        # type: () -> bool
        return self.is_source_missing() or self.is_source_incorrect()
