#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import math

from shinken.exceptions.business import ShinkenExceptionValueError
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, List


class Paginator:
    def __init__(self, page_size):
        # type: (int) -> None
        if page_size <= 0:
            raise ShinkenExceptionValueError(text=u'Wrong value:[ %s ] for page size parameter' % page_size)
        self._size = page_size
    
    
    def get_page_size(self):
        # type: () -> int
        return self._size
    
    
    def get_page(self, object_list, page_index):
        # type: (List[Any], int) -> List[Any]
        if page_index < 0:
            raise ShinkenExceptionValueError(text=u'Wrong value:[ %s ] for page number parameter' % page_index)
        start = self._size * page_index
        return object_list[start: start + self._size]
    
    
    def get_available_page_number(self, object_list):
        # type: (List[Any]) -> int
        return int(math.ceil(float(len(object_list)) / float(self._size)))
