#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

import re
import itertools

from shinken.log import LoggerFactory
from shinken.misc.type_hint import TYPE_CHECKING
from .index import Index

if TYPE_CHECKING:
    from shinken.misc.type_hint import Dict, Optional, AbstractSet, Any, Set
    from synchronizer.dao.items import BaseItem

indexed_collection_logger = LoggerFactory.get_logger('IndexedCollection')

RE_PATTERN_TYPE = re.Pattern


class SEARCH_FILTER:
    IN = '$in'
    NOT_IN = '$nin'
    NOT_EQUAL = '$ne'
    EXISTS = '$exists'


class SEARCH_OPERAND:
    AND = '$and'
    OR = '$or'


class IndexedCollection:
    DISABLED_OPERATOR_IN_NON_INDEXED_COLLECTION = True
    
    
    def __init__(self, hash_key=None):
        # type: (Optional[str]) -> None
        super(IndexedCollection, self).__init__()
        self.counter = 0
        self.indexes = {}  # type: Dict[str, Index]
        self.inverse = {}  # type: Dict[int, int]
        self.data = {}  # type: Dict[int, BaseItem]
        self.data_key_indexes = set()  # type: Set[int]
        self.hash_key = hash_key
    
    
    def __iter__(self):
        return iter(self.data.values())
    
    
    def __len__(self):
        return len(self.data)
    
    
    def append(self, item):
        self._append(item)
    
    
    def extend(self, items):
        if items:
            for item in items:
                self._append(item)
    
    
    def remove(self, search):
        item = self._find_one(search)
        if not item:
            raise KeyError('Item not found [%s]' % search)
        item_key = self._get_item_key(item)
        key_index = self.inverse[item_key]
        for index in iter(self.indexes.values()):
            index.unindex_item(key_index)
        del self.inverse[item_key]
        del self.data[key_index]
        self.data_key_indexes.discard(key_index)
    
    
    def regenerate_index(self, item):
        key_index = self.inverse[self._get_item_key(item)]
        for index in iter(self.indexes.values()):
            index.unindex_item(key_index)
            index.index_item(item, key_index)
    
    
    def add_index(self, property_name, getter=None, on_list=False):
        self.indexes[property_name] = Index(property_name, getter, on_list)
        self.indexes[property_name].add_index(self.data)
    
    
    def find_one(self, search):
        return self._find_one(search)
    
    
    def find(self, search, only_count=False):
        return self._find(search, only_count)
    
    
    def count(self, search):
        return self._find(search, only_count=True)
    
    
    def _get_item_key(self, item):
        if hasattr(item, 'get_key'):
            key = item.get_key()
        elif self.hash_key:
            key = item[self.hash_key]
        else:
            key = id(item)
        return key
    
    
    def _append(self, item):
        # type: (BaseItem) -> int
        key_index = self.counter
        self.data[key_index] = item
        self.data_key_indexes.add(key_index)
        item_key = self._get_item_key(item)
        self.inverse[item_key] = key_index
        for index in iter(self.indexes.values()):
            index.index_item(item, key_index)
        self.counter += 1
        return key_index
    
    
    def _find(self, search, only_count=False):
        # This method is used to remap result of _find_and_return_key_indexes to item or count
        if not search:
            return len(self.data) if only_count else list(self.data.values())
        
        found_key_indexes = self._find_and_return_key_indexes(search, already_filtered_indexes=self.data_key_indexes)
        if only_count:
            return len(found_key_indexes)
        
        found_items = list(map(lambda i: self.data[i], found_key_indexes))
        return found_items
    
    
    # NOTE: already_filtered_indexes can be either a set or a frozenset
    def _find_and_return_key_indexes(self, search, already_filtered_indexes):
        # type: (Dict[str, Any], AbstractSet[int]) -> AbstractSet[int]
        # This method search in data with search parameter and return only item key indexes (id)
        if not search:
            return already_filtered_indexes
        
        if len(search) == 1:
            prop_name, value_search = next(iter(search.items()))
        else:
            prop_name = SEARCH_OPERAND.AND
            value_search = [{k: v} for k, v in search.items()]
        
        if prop_name == SEARCH_OPERAND.OR:
            value_search = list(value_search)
            if not value_search:
                raise ValueError(f'Empty list given for {SEARCH_OPERAND.OR} operator')
            
            value_search_iterator = iter(value_search)
            found_key_indexes = self._find_and_return_key_indexes(next(value_search_iterator), already_filtered_indexes)
            for value in value_search_iterator:
                # Do not use -= because we don't want to modify original set (self.data_key_indexes)
                already_filtered_indexes = already_filtered_indexes - found_key_indexes
                found_key_indexes |= self._find_and_return_key_indexes(value, already_filtered_indexes)
            
            return found_key_indexes
        elif prop_name == SEARCH_OPERAND.AND:
            value_search = list(value_search)
            if not value_search:
                raise ValueError(f'Empty list given for {SEARCH_OPERAND.AND} operator')
            value_search_iterator = iter(value_search)
            found_key_indexes = self._find_and_return_key_indexes(next(value_search_iterator), already_filtered_indexes)
            for value in value_search_iterator:
                found_key_indexes &= self._find_and_return_key_indexes(value, found_key_indexes)
            return found_key_indexes
        
        prop_with_index = prop_name in self.indexes
        
        if prop_with_index and isinstance(value_search, dict):
            # Optimisation: If there is only one key (that should be), return directly the result
            value_search_iterator = iter(value_search.items())
            search_operator, search_value_for_operator = next(value_search_iterator)
            found_key_indexes = self._find_filter_operator_with_index(prop_name, search_operator, search_value_for_operator, already_filtered_indexes)
            
            for search_operator, search_value_for_operator in value_search_iterator:
                found_key_indexes &= self._find_filter_operator_with_index(prop_name, search_operator, search_value_for_operator, found_key_indexes)
        elif prop_with_index and not isinstance(value_search, RE_PATTERN_TYPE):
            found_key_indexes = set(self.indexes[prop_name].find(value_search))
        elif prop_name.startswith('$'):
            raise NotImplementedError('Operator [%s] is not yet implemented' % prop_name)
        else:
            indexed_collection_logger.warning('The property [%s] is not indexed. This will affect the search speed.' % prop_name)
            found_key_indexes = set(already_filtered_indexes)
            for found_key_index in already_filtered_indexes:
                found_item = self.data[found_key_index]
                if not self._recursive_search(found_item, prop_name, value_search):
                    found_key_indexes.remove(found_key_index)
        return found_key_indexes
    
    
    def _find_filter_operator_with_index(self, prop_name, search_operator, search_value_for_operator, already_filtered_indexes):
        # type: (str, str, str, AbstractSet[int]) -> AbstractSet[int]
        if search_operator == SEARCH_FILTER.IN:
            # The $in will search equality for each search_lookup
            index = self.indexes[prop_name]
            found_key_indexes = set(itertools.chain.from_iterable(index.find(sub_search_lookup) for sub_search_lookup in search_value_for_operator))
        elif search_operator == SEARCH_FILTER.NOT_EQUAL:
            # The $ne will exclude the search lookup
            index = self.indexes[prop_name]
            key_indexes_to_exclude = index.find(search_value_for_operator)
            
            # In case of multiple values for one property, $ne should match when the property contains all the values.
            if index.on_list:
                key_indexes_to_exclude = set(item_index for item_index in key_indexes_to_exclude if self.data[item_index].get(prop_name, '') == search_value_for_operator)
            
            found_key_indexes = already_filtered_indexes - key_indexes_to_exclude
        elif search_operator == SEARCH_FILTER.NOT_IN:
            # The $nin will exclude each the search lookup
            found_key_indexes = set(already_filtered_indexes)
            index = self.indexes[prop_name]
            for sub_search_lookup in search_value_for_operator:
                found_key_indexes -= index.find(sub_search_lookup)
        elif search_operator == SEARCH_FILTER.EXISTS:
            # The $exists will exclude the search lookup
            index = self.indexes[prop_name]
            items_with_values = set(index.reverse_index)
            if search_value_for_operator:
                found_key_indexes = items_with_values
            else:
                found_key_indexes = already_filtered_indexes - items_with_values
        else:
            raise NotImplementedError('Unknown filter operator [ %s ]' % (search_operator,))
        return found_key_indexes
    
    
    def _find_one(self, search):
        items = list(self._find(search))
        nb_items = len(items)
        if nb_items == 1:
            return items[0]
        elif nb_items > 1:
            raise ValueError('Too many value found')
        return None
    
    
    def _recursive_search(self, sub_dict, full_prop_name, value_search):
        full_prop_name = full_prop_name.split('.', 1)
        prop = full_prop_name[0]
        if len(full_prop_name) > 1:
            sub_props = full_prop_name[1]
            if sub_dict.get(prop, None):
                return self._recursive_search(sub_dict[prop], sub_props, value_search)
            else:
                return False
        else:
            if isinstance(value_search, dict):
                for search_key, search_lookup in value_search.items():
                    if search_key == SEARCH_FILTER.NOT_EQUAL:
                        return sub_dict.get(prop, '') != search_lookup
                    elif IndexedCollection.DISABLED_OPERATOR_IN_NON_INDEXED_COLLECTION:
                        raise NotImplementedError('[%s] filter is not implement in non-indexed collections due to performance' % value_search)
                    elif search_key in (SEARCH_FILTER.IN, SEARCH_FILTER.NOT_IN):
                        item_value_as_list = sub_dict.get(prop, '').split(',')
                        values_matches = set(search_lookup).intersection(item_value_as_list)
                        if (values_matches and search_key == SEARCH_FILTER.IN) or (not values_matches and SEARCH_FILTER.NOT_IN):
                            return True
                        else:
                            return False
                    elif search_key == SEARCH_FILTER.EXISTS:
                        item_has_field = prop in sub_dict
                        return search_lookup == item_has_field
                    else:
                        raise NotImplementedError('"%s" filter is not implement in %s find method' % (value_search, self.__class__.__name__))
            elif isinstance(value_search, RE_PATTERN_TYPE):
                return bool(value_search.search(sub_dict.get(prop, '')))
            else:
                return sub_dict.get(prop, '') == value_search
