#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2023:
# This file is part of Shinken Enterprise, all rights reserved.

import logging
import os

from shinken.log import FixedTimedRotatingFileHandler
from shinken.misc.type_hint import TYPE_CHECKING
from shinken.util import to_bool
from shinkensolutions.system_tools import set_ownership

if TYPE_CHECKING:
    from typing import Optional, Any
    from shinken.log import PartLogger
    from logging import Logger


class CreateUserLogFile:
    KEY_LOG_USERS__ENABLED = 'log_users__enabled'
    KEY_LOG_USERS__FILE_PATH = 'log_users__file_path'
    KEY_LOG_USERS__ADD_USER_NAME = 'log_users__add_user_name'
    KEY_LOG_USERS__ROTATION_TIME = 'log_users__logs_rotation__nb_days_before_deletion'
    
    
    def __init__(self, requester_prefix, conf, logger_error=None):
        #  type: (str, Any, Optional[PartLogger]) -> None
        self.log_enable = to_bool(getattr(conf, '%s__%s' % (requester_prefix, self.KEY_LOG_USERS__ENABLED), False))
        
        _default_path = '/var/log/shinken/%s/log_users.log' % requester_prefix
        self.log_file = getattr(conf, '%s__%s' % (requester_prefix, self.KEY_LOG_USERS__FILE_PATH), _default_path)
        if not self.log_file or not isinstance(self.log_file, str):
            self.log_file = _default_path
        
        self.rotation_number_of_days = int(getattr(conf, '%s__%s' % (requester_prefix, self.KEY_LOG_USERS__ROTATION_TIME), 7))
        # This logger are use for log errors of this logger,
        # classically use the default logger of the module or daemon
        self.logger_error = logger_error
    
    
    def create_path_and_add_right(self):
        # type: () -> None
        
        path_as_array = self.log_file.split('/')
        path_of_dirs = path_as_array[:-1]
        
        for i in range(1, len(path_of_dirs)):
            sub_path = '/'.join(path_as_array[:i + 1])
            if not os.access(sub_path, os.F_OK):
                try:
                    os.makedirs(sub_path)
                except Exception as e:
                    if self.logger_error:
                        self.logger_error.error('Could not create directories [%s] -> Error [%s]' % (sub_path, e))
                    return
                
                try:
                    set_ownership(sub_path)
                except Exception as e:
                    if self.logger_error:
                        self.logger_error.error('Could not set user right [%s] -> Error [%s]' % (sub_path, e))
                    return
        
        if not os.access(self.log_file, os.F_OK):
            try:
                with open(self.log_file, "a+") as f:
                    f.write('')
            except Exception as e:
                if self.logger_error:
                    self.logger_error.error('Could not create file [%s] -> Error [%s]' % (self.log_file, e))
                return
        
        if not os.access(self.log_file, os.W_OK):
            try:
                set_ownership(self.log_file)
            except Exception as e:
                if self.logger_error:
                    self.logger_error.error('Could not set user right [%s] -> Error [%s]' % (self.log_file, e))
                return
    
    
    def init_internal_logger(self):
        # type: () -> Logger
        self.create_path_and_add_right()
        try:
            logger = logging.getLogger('INTERNAL_LOGGER')
            if len(logger.handlers) == 0:
                log_handler = FixedTimedRotatingFileHandler(filename=self.log_file, when='midnight', backupCount=self.rotation_number_of_days)
                log_handler.setFormatter(logging.Formatter('%(message)s'))
                logger.addHandler(log_handler)
        except Exception as e:
            self.log_enable = False
            if self.logger_error:
                self.logger_error.error('Failed to open file [%s] -> Error [%s]' % (self.log_file, e))
            raise
        
        return logger
