#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2021:
# This file is part of Shinken Enterprise, all rights reserved.

from abc import ABCMeta, abstractmethod

import shinkensolutions.shinkenjson as json
from shinken.misc.type_hint import TYPE_CHECKING

if TYPE_CHECKING:
    from shinken.log import PartLogger
    from shinken.misc.type_hint import Any
    
    Path = str
    Lang = str
    TranslateKey = str


class TranslatePart(metaclass=ABCMeta):
    # No unicode here, the file is not in unicode, so we can't compare it with unicode
    COMMENT_CHAR = '//'
    START_FILE = 'var lang'
    END_FILE = '};'
    
    
    def __init__(self, logger: 'PartLogger', current_language: 'Lang', *, overridden_translations: 'dict|None' = None) -> None:
        self.overridden_translations = overridden_translations
        self.logger = logger
        self._current_language = current_language
        self._contents = {}
        self._contents_getter = None
        self._ = self.translate
        self._file_path = ''
        self._load()
        self._load_overridden_translations()
    
    
    def translate(self, translate_key, *args):
        # type: (TranslateKey, Any) -> str
        
        translation = self._contents_getter(translate_key, None)
        if translation is None:
            self.logger.error('Translate: cannot find %s in the lang %s%s' % (translate_key, self._current_language, ' in file : [%s]' % self._file_path if self._file_path else ''))
            return 'TO_TRAD(%s)' % translate_key
        try:
            _result = translation % args if args else translation
            return _result
        except TypeError as e:
            raise TypeError('Translate error : %s with trad_key [ %s ] and trad_args [ %s ]' % (e, translate_key, args))
    
    
    def is_translate_existing(self, translate_key):
        # type: (TranslateKey) -> bool
        return translate_key in self._contents
    
    
    def get_lang(self):
        # type: () -> Lang
        return self._current_language
    
    
    @abstractmethod
    def _load(self):
        raise NotImplementedError()
    
    
    def _load_overridden_translations(self) -> None:
        if not self.overridden_translations:
            return
        override_translations_flatten = {}
        self._flat_dict(self.overridden_translations, override_translations_flatten)
        
        for override_key, override_value in override_translations_flatten.items():
            if self._contents.get(override_key, ''):
                self._contents[override_key] = override_value
    
    
    def _read_dico_file(self, dico_file_path):
        # No str here, the file is not in unicode, so we can't compare it with unicode
        self._file_path = dico_file_path
        with open(dico_file_path, 'r') as f:
            buf = f.read()
        lines = buf.splitlines()
        new_lines = []
        for line in lines:
            line = line.strip()
            if line.startswith(self.COMMENT_CHAR):
                continue
            if line.startswith(self.START_FILE):
                line = '{'
            if line.startswith(self.END_FILE):
                line = '}'
            new_lines.append(line)
        buf = '\n'.join(new_lines)
        try:
            raw_trad = json.loads(buf)
        except ValueError as e:
            raise ValueError('bad JSON file structure, %s in %s ' % (e, f.name))
        flat_trad = {}
        self._flat_dict(raw_trad, flat_trad)
        return flat_trad
    
    
    def _flat_dict(self, node, flat_trad_dico, key_name=''):
        _key_name = ''
        for k, v in node.items():
            if key_name:
                _key_name = '.'.join([key_name, k])
            else:
                _key_name = k
            
            if isinstance(v, dict):
                self._flat_dict(v, flat_trad_dico, _key_name)
            else:
                flat_trad_dico[_key_name] = v


class SimpleTranslatePart(TranslatePart):
    @abstractmethod
    def _get_dico_file_path(self):
        # type: () -> Path
        raise NotImplementedError()
    
    
    def _load(self):
        dico_file_path = self._get_dico_file_path()
        if dico_file_path:
            flat_trad = self._read_dico_file(dico_file_path)
        else:
            flat_trad = {}
        
        self._contents = flat_trad
        self._contents_getter = flat_trad.get
