#!/usr/bin/python
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from shinken.toolbox.pickledb import ShinkenPickleableMeta

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, ShinkenIterable, Union, Dict, List, Optional

BOOLEAN_STATES = ('0', '1', True, False)

NUMBER_MAX_VALUE = 1000000000000000000000

UNAUTHORIZED_COLOR_CHARACTER_FOR_CSS = ['"', ';', '\'']


class ValidationMessages(object, metaclass=ShinkenPickleableMeta):
    KEYS_WARNINGS = 'warnings'
    KEYS_ERRORS = 'errors'
    KEYS_CRITICAL = 'critical'
    
    
    def __init__(self, warnings=None, errors=None, critical=None):
        # type: (Optional[List[str]], Optional[List[str]], Optional[List[str]]) -> None
        self._warnings = [] if warnings is None else warnings
        self._errors = [] if errors is None else errors
        self._critical = [] if critical is None else critical
    
    
    @staticmethod
    def from_dict(messages):
        # type: (Dict[str, List[str]]) -> ValidationMessages
        warnings = messages.get(ValidationMessages.KEYS_WARNINGS, [])
        errors = messages.get(ValidationMessages.KEYS_ERRORS, [])
        critical = messages.get(ValidationMessages.KEYS_CRITICAL, [])
        return ValidationMessages(warnings, errors, critical)
    
    
    def as_dict(self):
        # type: () -> Dict[str, List[str]]
        return {
            self.KEYS_WARNINGS: self._warnings,
            self.KEYS_ERRORS  : self._errors,
            self.KEYS_CRITICAL: self._critical
        }
    
    
    def add_warning(self, message):
        # type: (str) -> None
        self._warnings.append(message)
    
    
    def add_error(self, message):
        # type: (str) -> None
        self._errors.append(message)
    
    
    def add_critical(self, message):
        # type: (str) -> None
        self._critical.append(message)
    
    
    def get_nb_critical(self):
        # type: () -> int
        return len(self._critical)
    
    
    def get_nb_error(self, include_critical=False):
        # type: (bool) -> int
        return len(self._errors) + (self.get_nb_critical() if include_critical else 0)
    
    
    def get_nb_warning(self):
        # type: () -> int
        return len(self._warnings)
    
    
    def has_critical(self):
        # type: () -> bool
        return bool(self.get_nb_critical())
    
    
    def has_error(self, include_critical=False):
        # type: (bool) -> bool
        return bool(self.get_nb_error(include_critical=include_critical))
    
    
    def has_warning(self):
        # type: () -> bool
        return bool(self.get_nb_warning())
    
    
    def __add__(self, other):
        if not isinstance(other, type(self)):
            return NotImplemented
        self_copy = self.__class__(warnings=list(self._warnings), errors=list(self._errors), critical=list(self._critical))
        self_copy.__iadd__(other)
        return self_copy
    
    
    def __iadd__(self, other):
        if not isinstance(other, type(self)):
            return NotImplemented
        self._warnings.extend(other._warnings)
        self._errors.extend(other._errors)
        self._critical.extend(other._critical)
        return self


class ItemValidator:
    # Generic
    @staticmethod
    def _is_in_range(min_range, value, max_range, included):
        # type: (Any, Any, Any, bool) -> bool
        if min_range > max_range:
            raise ValueError('Min range cannot be greater than max range')
        return min_range <= value <= max_range if included else min_range < value < max_range
    
    
    @staticmethod
    def _is_positive(value, strict=False):
        # type: (Any, bool) -> bool
        return value > 0 if strict else value >= 0
    
    
    # Boolean validation
    @staticmethod
    def is_valid_boolean(value):
        # type: (Any) -> bool
        if isinstance(value, float):
            return False
        return value in BOOLEAN_STATES
    
    
    # Boolean validation
    @staticmethod
    def is_strict_json_boolean(value):
        # type: (Any) -> bool
        if value == "true":
            return True
        if value == "false":
            return True
        return isinstance(value, bool)
    
    
    # Integer validation
    @staticmethod
    def is_valid_integer(value):
        # type: (Union[str, int]) -> bool
        if value is True or value is False:
            return False
        if isinstance(value, str):
            try:
                value = int(value)
            except ValueError:
                return False
        
        if not isinstance(value, int):
            return False
        return value <= NUMBER_MAX_VALUE
    
    
    @staticmethod
    def is_integer_positive(value, strict=False):
        # type: (Union[str, int], bool) -> bool
        if not ItemValidator.is_valid_integer(value):
            return False
        
        return ItemValidator._is_positive(int(value), strict)
    
    
    @staticmethod
    def is_integer_in_range(min_range, value, max_range, included=True):
        # type: (int, Union[str, int], int, bool) -> bool
        if not ItemValidator.is_valid_integer(value):
            return False
        value = int(value)
        return ItemValidator._is_in_range(min_range, int(value), max_range, included)
    
    
    # Float validation
    @staticmethod
    def is_valid_float(value, cast_integer=False):
        # type: (Union[str, int, float], bool) -> bool
        if value is True or value is False:
            return False
        if isinstance(value, str) or (cast_integer and isinstance(value, int)):
            try:
                value = float(value)
            except ValueError:
                return False
        
        if not isinstance(value, float):
            return False
        
        return value <= NUMBER_MAX_VALUE
    
    
    @staticmethod
    def is_float_positive(value, cast_integer=True, strict=False):
        # type: (Union[float, int], bool, bool) -> bool
        if not ItemValidator.is_valid_float(value, cast_integer):
            return False
        return ItemValidator._is_positive(float(value), strict)
    
    
    @staticmethod
    def is_float_in_range(min_range, value, max_range, cast_integer=False, included=True):
        # type: (Union[int, float], Union[str, float,int], Union[float, int], bool, bool) -> bool
        if not ItemValidator.is_valid_float(value, cast_integer):
            return False
        value = float(value)
        return ItemValidator._is_in_range(min_range, value, max_range, included)
    
    
    # Enum validation
    @staticmethod
    def is_valid_list(enum):
        # type: (Any) -> bool
        return isinstance(enum, list)
    
    
    @staticmethod
    def is_valid_dict(enum):
        # type: (Any) -> bool
        return isinstance(enum, dict)
    
    
    @staticmethod
    def is_valid_enum_value(value, enum):
        # type: (Any, ShinkenIterable) -> bool
        return value in enum
    
    
    # Misc validation
    @staticmethod
    def is_valid_integer_or_is_valid_enum_value(value, enum):
        # type: (Any, ShinkenIterable) -> bool
        return ItemValidator.is_valid_integer(value) or ItemValidator.is_valid_enum_value(value, enum)
    
    
    @staticmethod
    def is_valid_positive_integer_or_is_valid_enum_value(value, enum, strict=False):
        # type: (Any, ShinkenIterable, bool) -> bool
        return (ItemValidator.is_valid_integer(value) and ItemValidator.is_integer_positive(value, strict)) or ItemValidator.is_valid_enum_value(value, enum)
    
    
    @staticmethod
    def is_valid_color_for_css(text):
        # type: (str) -> bool
        if not isinstance(text, str) or text == "":
            return False
        return not [c for c in UNAUTHORIZED_COLOR_CHARACTER_FOR_CSS if c in text]
    
    
    @staticmethod
    def is_string(text):
        # type: (str) -> bool
        return isinstance(text, str)
