#!/usr/bin/python-shinken
# -*- coding: utf-8 -*-
#
# Copyright (C) 2013-2024
# This file is part of Shinken Enterprise, all rights reserved.

from shinken.misc.type_hint import TYPE_CHECKING
from ....utils.weather_tools import travel_dictionaries, change_existing_value_if_equals_to

if TYPE_CHECKING:
    from shinken.misc.type_hint import Any, Dict, List, Optional


# WEATHER CONSTANTS ARE FORBIDDEN HERE

def _parse_existing(grid, path, default_value='default'):
    # type: (Dict, str, Optional[Any]) -> Optional[Any]
    return travel_dictionaries(grid, path, default_value=default_value)


def _safe_integer_conversion(value):
    # type: (str) -> str
    if value in ['default', 'all']:  # Special case -> Specific values that are not integers
        return value
    try:
        integer_value = int(value)
    except (TypeError, ValueError):  # Conversion could not be done, keep the actual value
        return value
    return str(integer_value)


def generate_new_grids(weather_grids):
    # type: (List[Dict]) -> List[Dict]
    
    to_return = []
    for grid in weather_grids:
        to_return.append({
            'grid_elements'     : _parse_grid_elements(grid) or _parse_existing(grid, 'grid_elements', []),
            'nb_tiles_in_height': _parse_nb_tiles_in_height(grid) or _parse_existing(grid, 'nb_tiles_in_height'),
            'nb_tiles_in_width' : _parse_nb_tiles_in_width(grid) or _parse_existing(grid, 'nb_tiles_in_width'),
            'layouts'           : _parse_grid_layouts(grid) or _parse_existing(grid, 'layouts', {}),
            'type'              : _parse_grid_type(grid) or _parse_existing(grid, 'type', 'grid')
        })
    return to_return


def _parse_grid_elements(grid):
    # type: (Dict) -> List
    to_return = []
    for grid_element in _parse_existing(grid, 'grid_elements', []):
        widget_type = _parse_existing(grid_element, 'content.type')
        to_return.append({
            'position_x': grid_element['position_x'],
            'position_y': grid_element['position_y'],
            'width'     : _safe_integer_conversion(_parse_existing(grid_element, 'width')),
            'height'    : _safe_integer_conversion(_parse_existing(grid_element, 'height')),
            'content'   : _parse_widget_content(widget_type, grid_element)
        })
    
    return to_return


def _parse_ordering(grid):
    # type: (Dict) -> str
    to_return = ''
    # if weather_version == 1:
    #     to_return = grid[WEATHER_CONSTS.WEATHER__GRIDS__ORDERING]
    
    return to_return


def _parse_nb_tiles_in_height(grid):
    # type: (Dict) -> str
    to_return = ''
    # if weather_version == 1:
    #     to_return = grid[WEATHER_CONSTS.WEATHER__GRIDS__NB_TILES_IN_HEIGHT]
    
    return to_return


def _parse_nb_tiles_in_width(grid):
    # type: (Dict) -> str
    to_return = ''
    # if weather_version == 1:
    #     to_return = grid[WEATHER_CONSTS.WEATHER__GRIDS__NB_TILES_IN_WIDTH]
    
    return to_return


def _parse_grid_layouts(grid):
    # type: (Dict) -> Dict
    return {
        'weather_widget_layout'  : _parse_weather_widget_layouts(grid),
        'title_widget_layout'    : _parse_title_widget_layouts(grid),
        'separator_widget_layout': _parse_separator_widget_layouts(grid)
    }


def _parse_grid_type(grid):
    # type: (Dict) -> str
    to_return = ''
    # if weather_version == 1:
    #     to_return = grid[WEATHER_CONSTS.WEATHER__GRIDS__TYPE]
    
    return to_return


_BACKGROUND_VALUES_RENAMES = {
    'from_webui': 'status_context_from_webui',
    'none'      : 'no_background'
}

_STATE_ICON_VALUES_RENAMES = {
    'from_webui': 'status_context'
}


def _replace_value_according_to_dict(value, reference_dict):
    # type: (str, Dict[str, str]) -> str
    
    for former_value, new_value in reference_dict.items():
        value = change_existing_value_if_equals_to(value, former_value, new_value)
    
    return value


def _parse_existing_layout(grid, widget_type, dotted_keys_from_layout, default_value='default'):
    # type: (Dict, str, str, Any) -> Any
    """
    Due to some magical happening thing, from Weather format 1 to 2,
    some weathers already have the 'layout' field whereas others have the former field 'default_values_for_grid_elements' :)
    """
    dotted_keys_format_1 = 'default_values_for_grid_elements.%s' % widget_type
    dotted_keys_format_2 = 'layouts.%s_layout' % widget_type
    
    return _parse_existing(
        grid,
        '.'.join([dotted_keys_format_2, dotted_keys_from_layout]),
        _parse_existing(grid, '.'.join([dotted_keys_format_1, dotted_keys_from_layout]), default_value)
    )


def _parse_weather_widget_layouts(grid):
    # type: (Dict) -> Dict
    widget_type = 'weather_widget'
    return {
        'width'  : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, 'width')),
        'height' : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, 'height')),
        'content': {
            'layout': {
                'background'            : _replace_value_according_to_dict(_parse_existing_layout(grid, widget_type, 'content.background'), _BACKGROUND_VALUES_RENAMES),
                'information_to_display': {
                    'left_area'     : {
                        'icon_type': _replace_value_according_to_dict(_parse_existing_layout(grid, widget_type, 'content.state_icon'), _STATE_ICON_VALUES_RENAMES)
                    },
                    'top_right_area': {
                        'displayed': 'default',
                        'icon_type': 'default'
                    },
                    'sla_area'      : {
                        'displayed': _parse_existing_layout(grid, widget_type, 'content.display_sla'),
                        'show_icon': 'default'
                    },
                    'name_area'     : {
                        'property_used_as_name': change_existing_value_if_equals_to(_parse_existing_layout(grid, widget_type, 'content.property_used_as_name'), 'definition_name', 'item_name'),
                        'nb_lines_max'         : 'default',
                        'vertical_align'       : 'default',
                        'font_zoom_percent'    : 'default'
                    }
                }
            }
        }
    }


def _parse_title_widget_layouts(grid):
    # type: (Dict) -> Dict
    widget_type = 'title_widget'
    return {
        'width'  : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, 'width')),
        'height' : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, 'height')),
        'content': {
            'layout': {
                'information_to_display': {
                    'title_area': {
                        'font_zoom_percent': 'default',
                        'text_color'       : _parse_existing_layout(grid, widget_type, 'content.text_color')
                    }
                }
            }
        }
    }


def _parse_separator_widget_layouts(grid):
    # type: (Dict) -> Dict
    widget_type = 'separator_widget'
    return {
        'width'  : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, 'width')),
        'height' : _safe_integer_conversion(_parse_existing_layout(grid, widget_type, 'height')),
        'content': {
            'layout': {
                'information_to_display': {
                    'left_area' : {
                        'displayed'    : 'default',
                        'width_percent': _parse_existing_layout(grid, widget_type, 'content.start_zone_text_percent')
                    },
                    'title_area': {
                        'text_color'       : _parse_existing_layout(grid, widget_type, 'content.text_color'),
                        'text_align'       : _parse_existing_layout(grid, widget_type, 'content.text_align'),
                        'font_zoom_percent': 'default'
                    },
                    'right_area': {
                        'displayed'    : 'default',
                        'width_percent': _parse_existing_layout(grid, widget_type, 'content.end_zone_text_percent')
                    }
                }
            }
        }
    }


def _parse_widget_content(widget_type, grid_element):
    # type: (str, Dict) -> Dict
    
    to_return = {
        'type'  : widget_type,
        'layout': _parse_widget_content_layout(widget_type, grid_element) or _parse_existing(grid_element, 'content', {})
    }
    if widget_type == 'weather_widget':
        to_return['item'] = _parse_existing(grid_element, 'content.item', {})
    
    return to_return


def _parse_widget_content_layout(widget_type, grid_element):
    # type: (str, Dict) -> Dict
    to_return = {}
    if widget_type == 'title_widget':
        to_return = _parse_title_widget_content_layout(grid_element)
    elif widget_type == 'separator_widget':
        to_return = _parse_separator_widget_content_layout(grid_element)
    elif widget_type == 'weather_widget':
        to_return = _parse_weather_widget_content_layout(grid_element)
    
    return to_return


def _parse_title_widget_content_layout(grid_element):
    # type: (Dict) -> Dict
    return {
        'information_to_display': {
            'title_area': {
                'label'            : _parse_existing(grid_element, 'content.title'),
                'text_color'       : _parse_existing(grid_element, 'content.text_color'),
                'font_zoom_percent': 'default'
            }
        }
    }


def _parse_separator_widget_content_layout(grid_element):
    # type: (Dict) -> Dict
    return {
        'information_to_display': {
            'left_area' : {
                'width_percent': _parse_existing(grid_element, 'content.start_zone_text_percent')
            },
            'title_area': {
                'text_color': _parse_existing(grid_element, 'content.text_color'),
                'text_align': _parse_existing(grid_element, 'content.text_align'),
                'label'     : _parse_existing(grid_element, 'content.title')
            },
            'right_area': {
                'width_percent': _parse_existing(grid_element, 'content.end_zone_text_percent')
            }
        }
    }


def _parse_weather_widget_content_layout(grid_element):
    # type: (Dict) -> Dict
    return {
        'background'            : _replace_value_according_to_dict(_parse_existing(grid_element, 'content.background'), _BACKGROUND_VALUES_RENAMES),
        'information_to_display': {
            'left_area': {
                'icon_type': _replace_value_according_to_dict(_parse_existing(grid_element, 'content.state_icon'), _STATE_ICON_VALUES_RENAMES)
            },
            'sla_area' : {
                'displayed': _parse_existing(grid_element, 'content.display_sla'),
            },
            'name_area': {
                'property_used_as_name': change_existing_value_if_equals_to(_parse_existing(grid_element, 'content.property_used_as_name'), 'definition_name', 'item_name'),
            }
        }
    }
